/*
 *      Qume Sprint 5 print filter
 *
 *      to do:  s_flag
 */

#include <stdio.h>
#include <signal.h>
#include <sys/types.h>
#include <sys/stat.h>

			 /********************************************/
#define SPACE     040    /* SP                                       */
#define BACK_SPC  010    /* BS                                       */
#define TAB       011    /* HT                                       */
#define NEWLINE   012    /* LF (=> CR LF)                            */
#define LINEFEED  013    /* VT (=> LF)                               */
#define FORMFEED  014    /* FF                                       */
#define CARG_RTN  015    /* CR                                       */
#define BOLD      016    /* SO     for shift to bold font            */
#define ROMAN     017    /* SI     for shift to roman font           */
#define ESCAPE    033    /* ESC    for special control sequences     */
#define NEG_LF    013    /* ESC VT for reverse line feed (=> ESC LF) */
#define POS_HLF   'U'    /* ESC U  for forward half line             */
#define NEG_HLF   'D'    /* ESC D  for reverse half line             */
#define CENTSIGN  ' '    /* ESC SP for printing cent sign            */
#define NOT_SIGN  '/'    /* ESC /  for printing not sign             */
#define GRAPHICS  'G'    /* ESC G  for fine graphics                 */
#define GRAPH_2   '3'    /* ESC 3  for dull graphics                 */
#define NORMAL    '4'    /* ESC 4  for non  graphics                 */
#define DEL      0177    /* DEL    for printable delimiter           */
#define SMALL_BS 0001    /* escaped BS for representing unit size BS */
#define SIZE     4096    /* size of array string                     */
			 /********************************************/
char **page;             /* page[0:length-1][0:wdth[row]-1]          */
int *wdth;               /* array of widths of each row              */
int *strt;               /* array of starting columns for each row   */
int *stop;               /* array of stopping columns for each row   */
int irow, icol;          /* position of next input character         */
int width  = 102;        /* page width,  default 102 spaces/line     */
int length =  66;        /* page length, default  66 lines/page      */
int offset =   0;        /* page offset, default   0 spaces          */
int hcpi = 12, vlpi = 6; /* columns/lines per inch, default 12/6     */
int s_flag = 0;          /* stop after each n pages, default no stop */
char font;               /* current font = ROMAN or BOLD             */
int hwm;                 /* high water mark, lowest line entered     */
int hsi, vsi;            /* current hor/ver space increments         */
int hci;                 /* current horizontal space for printable c */
int hinc, vinc;          /* non-graphics hor/ver space increments    */
int tabspace;            /* horizontal tab space increment           */
int vinc2;               /* vertical 1/2 line space increment        */
char string[SIZE];       /* input string being read                  */
int p;                   /* length of string (position of next char) */
int start;               /* column in which string starts            */
int srow, scol;          /* position of next character in string     */
int Irow, Srow;          /* irow/srow relative to start, not page    */
int max_row;             /* used to indicate if output < one page    */
int vertical;            /* size of line feed pending                */
int l_margin;            /* position of left margin                  */
int ocol;                /* position of next output character        */
int statflg = 0;         /* 1 if a tty, and able to chmod 600        */
char *tty;               /* name of stdout tty                       */
struct stat stat_info;   /* holds info on tty                        */
char *ttyname();         /* gets name of stdout tty                  */
int sig();               /* signal catching routine                  */
char *malloc();          /* standard dynamic memory allocation       */
char *realloc();         /* standard dynamic memory re-allocation    */
extern char *cmdname;    /* invoking name of this command            */
			 /********************************************/

main(argc, argv)
int argc;
char *argv[];
{
	FILE *fp;

	argc--;
	argv++;

	while (argc && argv[0][0] == '-') {
		switch (argv[0][1]) {
		case 'w':
			width  = atoi(*argv+2);
			break;
		case 'l':
			length = atoi(*argv+2);
			break;
		case 'o':
			offset = atoi(*argv+2);
			break;
		case 'h':
			hcpi   = atoi(*argv+2);
			break;
		case 'v':
			vlpi   = atoi(*argv+2);
			break;
		case 's':
			s_flag = atoi(*argv+2);
			if (s_flag == 0)
				s_flag++;
			break;
		case 'p':
			argc--;
			argv++;
			hcpi   = atoi(*argv);
			break;
		default:
			error("invalid %s option\n", *argv);
			break;
		}
		argc--;
		argv++;
	}

	hinc     = 120/hcpi;
	vinc     =  48/vlpi;
	offset  *= hinc;
	width   *= hinc;
	length  *= vinc;
	tabspace = hinc*8;
	vinc2    = vinc/2;
	l_margin = (offset>=0 ? offset : 0);
	ocol     = offset;

	if (signal(SIGINTR, SIG_IGN) == 0)
		signal(SIGINTR, sig);
	if (signal(SIGQUIT, SIG_IGN) == 0)
		signal(SIGQUIT, sig);
	signal(SIGTERM, sig);
	termset();
	printf("\033E%02d\033L%02d\033C%02d\0339",
		hinc, vinc, l_margin/hinc);

	page = (char **) malloc((unsigned) length*sizeof(char *));
	wdth = (int   *) malloc((unsigned) length*sizeof(int   ));
	strt = (int   *) malloc((unsigned) length*sizeof(int   ));
	stop = (int   *) malloc((unsigned) length*sizeof(int   ));
	if (page==NULL || wdth==NULL || strt==NULL || stop==NULL) {
		error("out of memory\n");
	}
	for (irow=0; irow<length; irow++) {
		page[irow] = malloc(0);
		if (page[irow]==NULL) {
			error("out of memory\n");
		}
		wdth[irow] = 0;
		stop[irow] = 0;
	}
	Irow     = 0;
	irow     = 0;
	icol     = offset;
	hwm      = length;
	p        = 0;
	Srow     = 0;
	srow     = 0;
	scol     = offset;
	start    = offset;
	font     = ROMAN;
	hsi      = hinc;
	hci      = hinc;
	vsi      = vinc;
	max_row  = 0;
	vertical = -1;

	if (argc) {
		while(argc) {
			fp = fopen(*argv, "r");
			if (fp == NULL) {
				error("cannot open %s\n", *argv);
			}
			qprf(fp);
			fclose(fp);
			argv++;
			argc--;
		}
	} else {
		if (isatty(fileno(stdin))) {
			error("no files specified\n");
		} else {
			qprf(stdin);
		}
	}

	flush();
	printf("\033C00\0339");
	termreset();
}

qprf(fp)
FILE *fp;
{
	int c;           /* current input character */

	for (;;) {
		switch(c = getc(fp)) {
		case SPACE:
			icol += hsi;
			break;
		case BACK_SPC:
			icol -= hsi;
			break;
		case TAB:
			c = icol%tabspace;
			if (c >= 0)
				icol += tabspace;
			icol -= c;
			break;
		case NEWLINE:
			inc_row(vsi);
			icol  = offset;
			break;
		case LINEFEED:
			inc_row(vsi);
			break;
		case FORMFEED:
			inc_row(length - irow);
			break;
		case CARG_RTN:
			icol  = offset;
			break;
		case BOLD:
		case ROMAN:
			font = c;
			break;
		case EOF:
			return;
			break;
		case ESCAPE:
			switch(c = getc(fp)) {
			case NEG_LF:
				inc_row(-vsi);
				break;
			case POS_HLF:
				inc_row(vinc2);
				break;
			case NEG_HLF:
				inc_row(-vinc2);
				break;
			case GRAPHICS:
				vsi = 1;
				hsi = 1;
				hci = 0;
				break;
			case GRAPH_2:
				vsi = 1;
				hsi = 2;
				hci = 0;
				break;
			case NORMAL:
				vsi = vinc;
				hsi = hinc;
				hci = hinc;
				break;
			case CENTSIGN:
			case NOT_SIGN:
				prntable(c + 0200);
				icol += hci;
				break;
			default:
				/* can't handle other control sequences */
				break;
			}
			break;
		default:
			if (SPACE <= c && c <= DEL) {
				prntable(c);
				icol += hci;
			}
			/* can't handle other control characters */
			break;
		}
	}
}

prntable(c)
int c;
{
	if (Srow != Irow) {
		enter();
	}
	while (scol >= icol+hinc) {
		string[p++] = BACK_SPC;
		scol -= hinc;
	}
	while (scol <  icol) {
		string[p++] = SPACE;
		scol += hinc;
	}
	while (scol > icol) {
		string[p++] = SMALL_BS;
		scol--;
	}
	if (font == BOLD) {
		string[p++] = c;
		string[p++] = BACK_SPC;
		string[p++] = c;
		string[p++] = BACK_SPC;
	}
	string[p++] = c;
	scol += hinc;
}

enter()
{
	int new_wid;
	char *new_row;
	int i, j;
	char *strg[2];
	int size[2];
	int frst[2];
	int last[2];
	int diff;
	char c;
	int spaces;
	int tics;
	int o, n;

	if (srow < 0)
		return;

	if (wdth[srow] == 0) {
		strt[srow] = start;
		stop[srow] = start;
	}
	o = strt[srow] > start;
	n = !o;
	strg[o] = page[srow];   strg[n] = string;
	size[o] = wdth[srow];   size[n] = p;
	frst[o] = strt[srow];   frst[n] = start;
	last[o] = stop[srow];   last[n] = scol;

	diff    = frst[1] - last[0];
	spaces  = diff / hinc;
	tics    = diff % hinc;
	if (diff < 0) {
		c = BACK_SPC;
		spaces = -spaces;
		tics   = -tics;
	} else {
		c = SPACE;
		if (tics > 0) {
			spaces++;
			tics = 10 - tics;
		}
	}
	new_wid = wdth[srow] + spaces + tics + p;
	new_row = malloc((unsigned) new_wid);
	if (new_row==NULL) {
		error("out of memory\n");
	}
	i = 0;

	for (j=0; j<size[0]; j++)
		new_row[i++] = strg[0][j];

	for (j=0; j<spaces; j++)
		new_row[i++] = c;

	for (j=0; j<tics; j++)
		new_row[i++] = SMALL_BS;

	for (j=0; j<size[1]; j++)
		new_row[i++] = strg[1][j];

	free(page[srow]);
	page[srow] = new_row;
	wdth[srow] = new_wid;
	strt[srow] = frst[0];
	stop[srow] = last[1];
	p     = 0;
	Srow  = Irow;
	srow  = irow;
	scol  = icol;
	start = icol;
}

inc_row(inc)
int inc;
{
	int nrow;
	static int underflo = 0;

	nrow = irow + inc;
	if (nrow > max_row)
		max_row = nrow;
	if (underflo && nrow > hwm)
		underflo = 0;
	if (!underflo) {
		if        (irow >= hwm && nrow <  hwm   ) {
			underflo = 1;

		} else if (irow <  hwm && nrow <  0     ) {
			nrow += length;
			underflo = nrow < hwm;

		} else if (irow <  hwm && nrow >= hwm   ) {
			while (nrow >= length) {
				emit (hwm, length-1);
				hwm   = 0;
				nrow -= length;
			}
			emit (hwm, nrow);
			hwm = nrow + 1;

		} else if (irow >= hwm && nrow >= length) {
			nrow -= length;
			if (nrow >= hwm) {
				while (nrow >= length) {
					emit (hwm, length-1);
					hwm   = 0;
					nrow -= length;
				}
				emit (hwm, nrow);
				hwm = nrow + 1;
			}
		}
	}
	irow = nrow;
	Irow += inc;
}

emit(from, upto)
int from, upto;
{
	int row;

	enter();
	for (row=from; row<=upto; row++) {
		vertical++;
		if (wdth[row] > 0) {
			output_v();
			output(row);
			page[row] = realloc(page[row], 0);
			wdth[row] = 0;
		}
	}
}

output(row)
int row;
{
	char *line;
	int col;
	int c;

	line = page[row];
	col  = strt[row];
	for (c=0; c<wdth[row]; c++) {
		switch (line[c]) {
		case SPACE:
			col += hinc;
			break;
		case BACK_SPC:
			col -= hinc;
			break;
		case SMALL_BS:
			col--;
			break;
		default:
			if (0<=col && col<width) {
				output_h(ocol, col);
				ocol = col + hinc;
				if (line[c] >= 0200) {
					putchar(ESCAPE);
					line[c] -= 0200;
				}
				putchar(line[c]);
			}
			col += hinc;
		}
	}
}

flush()
{
	enter();
	if (max_row < length)
		hwm = max_row + 1;
	else
		emit(hwm, length-1);
	emit(0, hwm-1);
	output_v();
}

output_h(here, there)
int here, there;
{
	int dist;

	if (there == here)
		return;

	for (dist=hinc; dist<=4*hinc; dist+=hinc) {
		if (there == here+dist) {
			putn(dist/hinc, SPACE);
			return;
		}
		if (there == here-dist) {
			putn(dist/hinc, BACK_SPC);
			return;
		}
		if (there == l_margin+(dist-hinc)) {
			putchar(CARG_RTN);
			putn(dist/hinc-1, SPACE);
			return;
		}
		if (there == l_margin-(dist-hinc)) {
			putchar(CARG_RTN);
			putn(dist/hinc-1, BACK_SPC);
			return;
		}
	}

	if (there % hinc == 0)
		printf("\033C%02d", there/hinc);
	else {
		dist = there - here;
		putchar(ESCAPE);
		putchar('H');
		if (dist > 0)
			putchar('@' + ((dist>>8)      ));
		else {
			dist = -dist;
			putchar('P' + ((dist>>8)      ));
		}
		putchar('@' + ((dist>>4) & 0xf));
		putchar('@' + ((dist   ) & 0xf));
	}
}

putn(n, c)
int n;
char c;
{
	int i;
	for (i=0; i<n; i++)
		putchar(c);
}

output_v()
{
	int lfs, hlfs;

	lfs  = vertical / 8;
	hlfs = vertical % 8 / 2;
	if (vertical%4 == 0 && lfs+2*hlfs < 5) {
		while (lfs-- > 0)
			putchar(LINEFEED);
		if (hlfs > 0) {
			putchar(ESCAPE);
			putchar('U');
		}
	} else {
		while (vertical > length) {
			putchar(ESCAPE);
			putchar('V');
			putchar('@' + ((length>>8)      ));
			putchar('@' + ((length>>4) & 0xf));
			putchar('@' + ((length   ) & 0xf));
			vertical -= length;
		}
		putchar(ESCAPE);
		putchar('V');
		putchar('@' + ((vertical>>8)      ));
		putchar('@' + ((vertical>>4) & 0xf));
		putchar('@' + ((vertical   ) & 0xf));
	}
	vertical = 0;
}

termset()
{
	tty = ttyname(1);
	if (tty != 0) {
	        if (stat(tty, &stat_info) == 0) {
		        statflg++;
		        chmod(tty, stat_info.st_mode & 0600);
	        } else {
		        error("can't stat %s\n", tty);
	        }
	}
}

termreset()
{
	if (statflg)
		chmod(tty, stat_info.st_mode & 0777);
}

sig(n)
{
	putchar('\n');
	psignal(cmdname, n);
	termreset();
	_exit(1);
}

error(arglist)
int *arglist;
{
	fprintf(stderr, "%s: %r", cmdname, &arglist);
	termreset();
	exit(1);
}
