#include <stdio.h>
#include "schart.h"

#define HYPHEN  '-'             /* hyphen */
#define EXCLUDE 'x'             /* exclude flag */
#define INCLUDE 'i'             /* include flag */

#define MAXSYM  200             /* max num of excluded symbols */

struct  node    branches;       /* original header node */
struct  node    *root;          /* pointer to root of current branch */
struct  node    *free_nodes = NULL;     /* free node ptr */
struct  node    *last_node = NULL;      /* end of free space */

int     nsym = 0;               /* number of exclude symbols */
char    symtab[MAXSYM][FNLEN+1];/* exclude symbol table */
char    *psym[MAXSYM];          /* sorted array of ptrs to symtab[] */
char    dsymfile[] = "/usr/lib/schart.excl";    /* default exclude file */


/*
*   Input is read from the argument files and transformed into a
*   tree data structure.  Each line of
*   input corresponds to a node of the tree.  A "definition" node
*   is a father and all its "calls" are sons.  Initially, a
*   very broad two level tree is created, with each of the
*   "definition" nodes connected as brothers.
*
*   After reading all the input, the broad tree is transformed
*   into a regular tree.  Beginning with the first "definition" node,
*   an in-order tree search is preformed.  When a leaf is found,
*   the "definition" nodes are searched to see if one exists with
*   the same name as the leaf.  If a match is found, the son of the
*   "definition" node becomes the son of the leaf.
*   The in-order search then continues with the new
*   son of the leaf.
*/


main(argc, argv)
        int argc;
        char *argv[ ];
{
	char    iflg;           /* include names in schart.excl */
	int     compar();       /* sort comparison routine */

        argc--;
	argv++;
	iflg = 0;
	while (**argv == HYPHEN) {
		argc--;
		switch(*++(*argv)) {
case EXCLUDE:           if (argc) {
			        argc--;
                                rdsyms(*++argv);
		        }
			break;

case INCLUDE:           iflg++;
			break;
		}
		argv++;
        }
	if (!iflg)
		rdsyms(dsymfile);
        qsort(psym, nsym, sizeof(psym[0]), compar);
        broadtree(argc, argv);
	exclude();
        buildtree();
        cleanup(&branches);
        schart(branches.n_bro);
}


/* Build a broad two level tree.
*  If command line arguments are present, open the files
*  and process them.  Otherwise, assume the input is coming
*  from the standard input.
*  Returns:  nothing.
*/
broadtree(argc, argv)
        int     argc;           /* number of args */
        char    **argv;         /* arg vector */
{
        FILE    *ioinptr;       /* file descriptor */

        if (argc) {
                while (argc) {
                        if ((ioinptr = fopen(*argv, "r")) == NULL) {
                                fprintf(stderr, "%s: cannot open\n",
				        *argv);
                                exit(1);
                        }
                        proc_file(ioinptr);
                        fclose(ioinptr);
                        argv++;
			argc--;
		}
        }
        else
                proc_file(stdin);
}


/* Process an input file.
*  Read the input lines.
*  If a line is a definition node, attach it to the brother chain
*  anchored by "branches".
*  If it is call node, then attach it as a son of the last
*  definition node.  If the last defintion node already has
*  a son with the same name ignore it.
*  Check for both duplicate definitions and calls.
*  Returns:  nothing.
*/
proc_file(fd)
        FILE    *fd;            /* file descriptor */
{
        struct  node    *p;     /* ptr to current definition node */
        struct  node    *r;     /* ptr to current call node */
        struct  node    *getnode();     /* function to get a new node */

        char    funcname[FNLEN+1];     /* function name */
        char    type[30];              /* type: either def or call */

        while (fscanf(fd, "%s %*s %s", funcname, type) != EOF) {
		funcname[FNLEN] = EOS;
                if (type[0] == 'd' ) {
                        p = &branches;
			while ((p->n_bro != NULL) && strcmp(funcname, p->n_bro->n_name))
				p = p->n_bro;
			if (p->n_bro == NULL)
				p = getnode(&p->n_bro, funcname);
			else
                                fprintf(stderr, "%s: Duplicately defined\n", funcname);
                }
		else if (type[0] == 'c') {
                        if (p->n_son == NULL)
                                r = getnode(&p->n_son, funcname);
                        else {
                                r = p->n_son;
                                while (strcmp(funcname, r->n_name)) {
                                        if (r->n_bro == NULL) {
                                                r = getnode(&r->n_bro, funcname);
                                                break;
                                        }
                                        r = r->n_bro;
                                }
                        }
                }
		else {
			fprintf(stderr, "Bad input format\n");
			exit(1);
		}
	}
}

/* rdsyms - read exclude symbols
*  Read the symbols from the given file and put them in the
*  symbol table
*  returns:  nothing.
*/
rdsyms(fn)
	char    *fn;
{
	FILE    *fd;            /* file descriptor */

	if ((fd = fopen(fn, "r")) == NULL) {
		fprintf(stderr, "%s: Unable to Open\n", fn);
		return;
	}
	while (nsym < MAXSYM) {
		if (fscanf(fd, "%8s", symtab[nsym]) == EOF) {
			fclose(fd);
			return;
		}
                psym[nsym] = symtab[nsym];
                nsym++;
	}
	fprintf(stderr, "Exclude symbol table overflow\n");
	fclose(fd);
}

/* exclude - exclude specified nodes from the tree
*  The names in the exclude symbol table are to be
*  removed from the tree.  However, if there is a definiton
*  node for a name it is to remain.  Look up all the definition
*  nodes in the symbol table and remove any matches from the symbol
*  table.  Then look up all the leafs of the two level tree and
*  remove any leaves that are also found in the symbol table.
*  returns:  nothing.
*/
exclude()
{
	int     n;
	struct  node    *p, *q, *r;

	if (nsym == 0)
		return;

	for (p = branches.n_bro; p; p = p->n_bro)
		if ((n = srch(psym, nsym, p->n_name)) >= 0)
                        *psym[n] = EOS;

	for (p = branches.n_bro; p; p = p->n_bro) {
		while (q = p->n_son) {
			if (srch(psym, nsym, q->n_name) >= 0)
				p->n_son = q->n_bro;
                        else  {
                                while (r = q->n_bro)
                                        if (srch(psym, nsym, r->n_name) >= 0)
                                                q->n_bro = r->n_bro;
                                        else
                                                q = r;
                                break;
                        }
		}
	}
}

/* compar - comparison routine for qsort()
*  returns: same as strcmp()
*/
compar(x, y)
	char    **x;
	char    **y;
{
	return(strcmp(*x, *y));
}

/* srch - binary srch routine for excluded symbol look up
*  returns:  an index into the pointer array of the found symbol,
*            -1 if the symbol not found.
*/
srch(psym, i, name)
	char    *psym[];
	int     i;
	char    *name;
{
	int     lo, hi, n;
	int     rc;

        lo = 0;
        hi = i - 1;
        do {
                n = (lo + hi)/2;
		rc = strcmp(name, psym[n]);
                if (rc == 0)
                        return(n);
                else if (rc < 0)
                        hi = n - 1;
                else
                        lo = n + 1;
        } while (lo <= hi);
	return(-1);
}

/* Build a regular tree from the broad tree.
*  Traverse the tree with an in-order search.
*  When a leaf is found, see if it can be extended by connecting
*  it to one of the definition nodes (the brother chain from "branches").
*  Returns: nothing.
*/
buildtree()
{

        root = branches.n_bro;
        while (root != NULL) {
                if (root->n_comp == TRUE && root->n_son != NULL)
                        traverse(root);
                root = root->n_bro;
        }
}


/* traverse - in-order tree traversal.
*  Traverse until a leaf is found.  Then, try to connect the
*  leaf.
*  Returns:  nothing.
*/
traverse(p)
	struct  node    *p;     /* the current node */
{

        if (p->n_son == NULL)
		leaf(p);
	if (p->n_son != NULL)
		traverse(p->n_son);
	if (p->n_bro != NULL && p != root)
		traverse(p->n_bro);
}


/* leaf - Try to connect a leaf node to a definition node.
*  Compare the function name of the current node with the function
*  names of the definition nodes.
*  If there is a match, set the son pointer of the current node
*  to the son pointer of the definition node.  Then mark the
*  definition node as incomplete and set its son pointer to the
*  current node.  This is for cross reference purposes.
*  Returns:  nothing.
*/
leaf(p)
        struct  node    *p;     /* the current node */
{
        struct  node    *q;     /* ptr to definition nodes */

        q = branches.n_bro;
        while (q != NULL && strcmp(p->n_name, q->n_name)) {
                q = q->n_bro;
        }
        if (q == NULL)
		return;
        if (q != root && q->n_comp == TRUE) {
		if (q->n_son)
			p->n_son = q->n_son;
		q->n_comp = FALSE;
	}
	else if (q->n_son)
		p->n_comp = FALSE;
}

/* getnode - Get a new node.
*  If free storage exists, carve the node out of it.
*  Otherwise, make a system call to get enough storage
*  for 100 more nodes.
*  Initialize the node.
*  Returns:  a pointer to the new node.
*/
struct node *
getnode(ptr, funcname)
	struct  node    **ptr;  /* ptr to location to link new node into */
        char    *funcname;      /* name of the new node */
{
        struct node *newnode;
	char    *malloc();

        if (free_nodes >= last_node) {
                free_nodes = (struct node *) malloc(100*sizeof(*newnode));
		if ((int) free_nodes == -1) {
			fprintf(stderr, "Out of storage\n");
			exit(1);
		}
                last_node = free_nodes + 100;
        }
        newnode = free_nodes;
        free_nodes++;
	strcpy(newnode->n_name, funcname);
        newnode->n_comp = TRUE;
        newnode->n_son = NULL;
        newnode->n_bro = NULL;
        *ptr = newnode;
        return(newnode);
}


/* Cleanup the tree.
*  Check each definition node for whether it is complete or not.
*  If it is complete leave it, otherwise prune it from the tree.
*  Returns:  nothing.
*/
cleanup(p)
        struct  node    *p;
{
        struct  node    *savep;

        savep = p;
        p = p->n_bro;
        while (p != NULL) {
                if (p->n_comp != TRUE)
			savep->n_bro = p->n_bro;
                else
			savep = p;
                p = p->n_bro;
        }
}
