/*
*  Synopsis : acctpk [-st]
*
*  This program gathers the rather lengthy accounting records data
*  from "acctx" or "accty" and fills some more compact summary
*  structures.  These structures are written out on a daily basis
*  to a file whose name is the julian date that the records were
*  created.
*
*  The original accounting records file may be truncated at the end
*  of the process by giving the -s flag.
*
*  A test of the acctpk routine may be made by giving the -t flag.
*  No data will actually be written out, and the -s flag is ignored
*  if given.
*/

#include "/usr/src/cmd/accounting/acct.h"
#include <sys/acctfile.h>
#include <pwd.h>
#include <sys/param.h>
#include <sys/filsys.h>
#include <sys/ino.h>
#include <utmp.h>
#include <acct.h>
#include <stdio.h>
#include <ctype.h>
#include <signal.h>
#include <time.h>

char dayfile[30];
char *acctf;
FILE *inacct;
int  lotime, hitime, endday, dtime;
int  currtim;
int  sflag, tflag;

/*
*  The following are used to calculate the time for the end-of-day.
*  "timezone" is the difference between the system's time (GMT) and the
*  local time.  It's value is kept by the 'ctime' call.
*/
#define DAYSECS  (24*60*60)
int  timezone;

/* General Constants */
#define MAXTTY   70
#define BYTEPWD  4
#define MAXDEV   (DAYSECS*7)

/* Input structures */
union {
	struct acct     type0;
	struct acct_log type1;
	struct acct_io  type234;
	struct {
		int  rid;
		char data[1];
	} gen;
} acctrec;

struct acct     acct;
struct acct_io  rd;
struct acct_io  pr;
struct acct_io  pu;
struct utmp     lo;

/* Output structures */
struct aday aday[MAXUSERS];
struct cmds cmds[MAXCMDS];

/* Array has the number of blocks used by each user */
int blocks[MAXUSERS];

/* Structures to tally connect time for each user */
struct ubuf {
	char xname[8];
	int  uutime;
} ubuf[MAXUSERS];

struct tbuf {
	struct ubuf *userp;
	char tline[8];
	int  ttime;
} tbuf[MAXTTY];


main(argc, argv) char **argv;
{
	int i, n, ignored, used, fflag;
	char *oldfile();
	struct tbuf *tp;
        struct acct_log cbuf;

	while(--argc > 0 && *argv[1]=='-') {
		argv++;
		while(*++*argv) switch(**argv) {
			case 's':
				sflag++;
				continue;
			case 't':
				tflag++;
				sflag = 0;
				continue;
			default:
				printf("usage: acctpk[-st]\n");
				exit(1);
		}
	}
	if (argc > 0) {
		printf("Usage: acctpk [-st]\n");
		exit(1);
	}

	/* decide whether to read from acctx or accty */
	acctf = oldfile();

	/* count current user blocks into "blocks"  */
	if (!tflag)
		countblk();

	/* get the first record */
	if ((inacct = fopen(acctf, "r")) == NULL  || getrec() == 0) {
		printf("Can't read %s file.\n", acctf);
		exit(1);
	}

	/* process each days records */
	for (;;) {
		setfile(currtim);
		init(currtim);
		ignored = used = fflag = 0;
		while(currtim < endday) {
			/* we will ignore any records older than the */
			/* last recorded "hitime"                    */
			if (currtim >= hitime || fflag) {
				switch(acctrec.gen.rid)  {
					case 0:
						acctacct();
						break;
					case 1:
						connacct();
						break;
					case 2:
						readacct();
						break;
					case 3:
						punacct();
						break;
					case 4:
						prtacct();
						break;
				}
				used++;
				fflag  = 1;
				hitime = currtim;
			}
			else
				ignored++;
			if ((n = getrec()) == 0)
				break;
		}
		/* put "blocks" and "ubuf" data into "aday" */
		blkload();
		connload(n<=0 ? currtim:endday);

		/* write that mothra out */
		if (!tflag)
			outday();
		printf("Output %s : %5d used, %5d ignored\n",
			dayfile, used, ignored);
		if (n <= 0) break;
	}
	fclose(inacct);

	/* now we can truncate the input file */
	if (sflag && !tflag)
		creat(acctf, 0644);

        /* do an acctwrt for users logged on at endday */
	for (tp=tbuf; tp<&tbuf[MAXTTY]; tp++) {
		if (tp->userp) {
                        cbuf.al_rid = 1;
                        for (i=0; i<8; i++) {
                                cbuf.al_utmp.ut_line[i] = tp->tline[i];
                                cbuf.al_utmp.ut_name[i] = tp->userp->xname[i];
                        }
                        cbuf.al_utmp.ut_time = currtim;
                        acctwrt(&cbuf, sizeof cbuf);
		}
	}
}


/*
* oldfile: looks in the file /etc/acctfile to determine which
*          file is the active file.  It then returns the address
*          of the other one.
*/
char *oldfile()
{
	char active[30];
	FILE *afd;

	if ((afd=fopen("/etc/acctfile", "r"))==NULL) {
		printf("Cannot open /etc/acctfile\n");
		exit(1);
	}
	fgets(active, 30, afd);
	fclose(afd);
	active[strlen(active)-1] = NULL;
	if (strcmp(active, filex)==0) return(filey);
	if (strcmp(active, filey)==0) return(filex);
	/* ? unknown file ? */
	printf("Invalid active acct file: %s\n", active);
	exit(1);
}

/*
* setfile: puts the julian date for the given time (in seconds)
*          into the filename "dayfile".
*/
setfile(attime)
{
	struct tm *tm;
	struct tm *localtime();

	tm = localtime(attime);
	sprintf(dayfile, daymodel, tm->tm_year*1000 + tm->tm_yday+1);
}


/*
* init : is a complex little fellow ... if the current dayfile
*        does not exist, it just initializes "lotime", "hitime",
*        "endday", "cmds" and "aday".  If "dayfile" does exist,
*        it treats that file as gospel and reloads the buffers.
*/
init(attime)
{
	int  n, users;
	FILE *tfd;

	/* figure out when midnight is */
	endday = attime/DAYSECS + 1;
	endday = endday*DAYSECS + timezone;

	lotime = hitime = attime;
	zeroout((int *)aday, sizeof aday);
	zeroout((int *)cmds, sizeof cmds);

	/* read existing user information */
	if ((tfd = fopen(dayfile, "r"))==NULL) {
/*              printf("Couldn't open %s\n", dayfile);/*DEBUG*/
		return;
	}

	lotime = getw(tfd);
	hitime = getw(tfd);
	users  = getw(tfd);
	if (lotime==EOF || hitime==EOF || users==EOF) {
		lotime = hitime = attime;
		fclose(tfd);
		printf("Problems reading %s\n", dayfile);
		return;
	}
	fread ((char *)aday, sizeof *aday, users, tfd);

	/* and get command usage stuff */
	while ((n = getw(tfd)) != EOF)
		fread((char *)&cmds[n], sizeof *cmds, 1, tfd);
	fclose(tfd);
}


/*
* getrec: gets the next record in the accounting file according
*         to the record-id.
*/
getrec()
{
	int n, inmess;

	inmess = 0;
nextrec:
	if (inmess) acctrec.gen.rid = getw(inacct);
readrec:
	if ((acctrec.gen.rid = getw(inacct)) == EOF)
		return(0);
	switch(acctrec.gen.rid) {
		case A_PROC:
			n = fread(acctrec.gen.data, sizeof acct - 4, 1, inacct);
                        acct = acctrec.type0;
			currtim = acct.ac_btime;
			acct.ac_comm[DIRSIZ-1] = 0;
			break;
		case A_LOGIN:
			n = fread(acctrec.gen.data, sizeof acctrec.type1 - 4, 1, inacct);
			lo = acctrec.type1.al_utmp;
			currtim = lo.ut_time;
			lo.ut_name[7] = 0;
			break;
		case A_READ:
			n = fread(acctrec.gen.data, sizeof rd - 4, 1, inacct);
			rd = acctrec.type234;
			currtim = rd.ai_ctime;
			break;
		case A_PUNCH:
			n = fread(acctrec.gen.data, sizeof pu - 4, 1, inacct);
			pu = acctrec.type234;
			currtim = pu.ai_ctime;
			break;
		case A_PRINT:
			n = fread(acctrec.gen.data, sizeof pr - 4, 1, inacct);
			pr = acctrec.type234;
			currtim = pr.ai_ctime;
			break;
		default:
			printf("Invalid record id = %d\n", acctrec.gen.rid);
			inmess++;
			/* try and find a valid one */
			goto readrec;
	}
	if (n != 1) return(0);
	if (checkrec() != 0) goto nextrec;
	return(1);
}


/*
* checkrec: makes sure that the retrieved record is not garbage, and
*           optionally prints the record.  If more than 10 records are
*           found as garbage, an exit is taken.
*/
checkrec()
{
	int errflg;
	static lasttime, errcnt;
	char *ctime();

	errflg = 0;
	if (lasttime)
		if (currtim-MAXDEV>lasttime || currtim+MAXDEV<lasttime)
			errflg++;
	if (!errflg)
		lasttime = currtim;

	/*
	*  This switch is conveniently placed here in case we have to do
	*  garbage checking based on record type.
	*/
	if (!errflg) switch(acctrec.gen.rid) {
		case 0:
			break;
		case 1:
			break;
		case 2:
			break;
		case 3:
			break;
		case 4:
			break;
	}

	if (tflag || errflg) switch(acctrec.gen.rid) {
		case 0:
			printf("acct: cmd=%-14.14s usr=%-4d cpu=%-6D elap=%-6d io=%-6d time=%s",
			acct.ac_comm,acct.ac_uid,acct.ac_stime+acct.ac_utime,acct.ac_etime,
			acct.ac_rdwr,ctime(acct.ac_btime));
			break;
		case 1:
			printf("lo: usr=%-8.8s tty=%s time=%s",
			lo.ut_name,lo.ut_line,ctime(lo.ut_time));
			break;
		case 2:
			printf("rd: usr=%-4d recs=%-8d time=%s",
			rd.ai_uid,rd.ai_recs,ctime(rd.ai_ctime));
			break;
		case 3:
			printf("pu: usr=%-4d recs=%-8d time=%s",
			pu.ai_uid,pu.ai_recs,ctime(pu.ai_ctime));
			break;
		case 4:
			printf("pr: usr=%-4d recs=%-8d time=%s",
			pr.ai_uid,pr.ai_recs,ctime(pr.ai_ctime));
			break;
	}
	if (errflg) {
		if (++errcnt > 50) {
			printf("Too many bad records\n");
			notify("adm", "Too many bad records", 1);
			exit(1);
		}
		return(1);
	}
	return(0);
}


/*
* outday : actually writes the stuff for one day out to the previously
*          determined "dayfile".
*/
outday()
{
	int  n, oldsig[4], users;
	FILE *tfd;

	/* see how many "real" users we have */
	for(n = MAXUSERS-1;
		n>0 && aday[n].ublk==0 && aday[n].uconn==0
		&& aday[n].rread==0 && aday[n].scpu==0
		&& aday[n].ucpu==0;
		n--) ;
	users = n+1;

	/* this is dangerous, so we don't want interuptions */
	for (n=1; n<4; n++)
		oldsig[n] = signal(n,SIG_IGN) == SIG_IGN;

	if ((tfd = fopen(dayfile, "w")) == NULL) {
		printf("Cannot create %s\n", dayfile);
		exit(1);
	}
	putw(lotime, tfd);
	putw(hitime, tfd);
	putw(users , tfd);
	fwrite((char*)aday, sizeof *aday, users, tfd);
	for (n=0; n<MAXCMDS; n++)
		if (cmds[n].count > 0) {
			putw(n, tfd);
			fwrite((char*)&cmds[n], sizeof *cmds, 1, tfd);
		}
	fclose(tfd);

	chmod(dayfile, 0644);
	for (n=1; n<4; n++)
		if (oldsig[n]==0) signal(n, SIG_DFL);
}

/*
* acctacct: handles all cpu and command usage accounting
*/
acctacct()
{
	int i, uid;

	/* change cpu times from microseconds to milliseconds */
	acct.ac_stime = acct.ac_stime / 1000;
	acct.ac_utime = acct.ac_utime / 1000;

	/* user accounting */
	uid = acct.ac_uid;
	if (uid < 0  ||  uid > MAXUSERS-1) uid = 0;
	aday[uid].nocmd++;
	aday[uid].scpu += acct.ac_stime;
	aday[uid].ucpu += acct.ac_utime;
	aday[uid].uio  += acct.ac_rdwr;

	/* command usage accounting */
	if ((i = enter(acct.ac_comm)) < 0) {
		printf("Command Table Overflow.\n");
		return;
	}
	cmds[i].count++;
	cmds[i].realt += acct.ac_etime;
	cmds[i].cput  += acct.ac_utime;
	cmds[i].syst  += acct.ac_stime;
	cmds[i].io    += acct.ac_rdwr ;
}


/*
* readacct: handles all VMREAD records accounting
*/
readacct()
{
	int uid;

	uid = rd.ai_uid;
	if (uid < 0  ||  uid > MAXUSERS-1) uid = 0;
	aday[uid].rread += rd.ai_recs;
}


/*
* punacct: handles all VMPUNCH records accounting
*/
punacct()
{
	int uid;

	uid = pu.ai_uid;
	if (uid < 0  ||  uid > MAXUSERS-1) uid = 0;
	aday[uid].rpun += pu.ai_recs;
}


/*
* prtacct: handles all OPR records accounting
*/
prtacct()
{
	int uid;

	uid = pr.ai_uid;
	if (uid < 0  ||  uid > MAXUSERS-1) uid = 0;
	aday[uid].rprt += pr.ai_recs;
}


/*
* connacct: handles all connect time accounting ... a royal pain, essay!
*/
connacct()
{
	int i, j, flag;
	struct tbuf *tp;

	/* need to adjust for reset of system time? */
	if (lo.ut_line[0] == '|') {
		dtime = lo.ut_time;
		return;
	}
	if (lo.ut_line[0] == '}') {
		if (dtime==0) return;
		i = lo.ut_time - dtime;
		for (tp=tbuf; tp<&tbuf[MAXTTY]; tp++) {
			tp->ttime += i;
		}
		dtime = 0;
		return;
	}

	/* was the system rebooted? */
	if (lo.ut_line[0] == '~') {
		lo.ut_name[0] = 0;
		for (tp=tbuf; tp<&tbuf[MAXTTY]; tp++)
			connup(tp, 0);
		return;
	}

	/* normal tty record, update info for that tty */
	/* find pointer */
	flag = 0;
	for (i=0; i<MAXTTY; i++)
		if (tbuf[i].tline[0] == NULL && !flag) {
			j = i;
			strncpy(tbuf[i].tline, lo.ut_line, 8);
			flag++;
		} else
		        if (strncmp(lo.ut_line, tbuf[i].tline, 8) == 0) {
				j = i;
				break;
			}
	tp = &tbuf[j];

	/* cleanup login name */
	flag = 0;
	for (i=0; i<8; i++) {
		if (isspace(lo.ut_name[i]) || lo.ut_name[i]==NULL) {
			flag++;
			lo.ut_name[i] = NULL;
		} else {
			if (flag)
				return;
		}
	}

	connup(tp, 0);
}


/*
* connup: updates the connect time info for a single tty (user)
*/
connup(tp, freeze) struct tbuf *tp;
{
	struct ubuf *up;
	int i, t, t1;

	/* if "freeze" is non-zero, the time for the given user is */
	/* calculated up to the freeze time, and the tty time is   */
	/* reset to that time.                                     */
	t = freeze ? freeze : lo.ut_time;
	if (tp->userp) {
		t1 = t - tp->ttime;
		if (t1>0)
			tp->userp->uutime += t1;
	}
	tp->ttime = t;
	if (freeze) return;

	/* was this a user logout for the tty? */
	if (lo.ut_name[0]==0) {
		tp->userp = 0;
		return;
	}

	/* some user signed on to the tty */
	for (up=ubuf; up<&ubuf[MAXUSERS]; up++) {
		if (up->xname[0]==0)
			break;
		for (i=0; i<8 && up->xname[i]==lo.ut_name[i]; i++) ;
		if (i>=8)
			break;
	}
	for (i=0; i<8; i++)
		up->xname[i] = lo.ut_name[i];
	tp->userp = up;
}


/*
* connload: freeze the connect time accounting at "endday", put the
*           counted-up connect time for each user into "aday", and
*           reset "ubuf" times to 0.
*/
connload(frtime)
{
	struct ubuf *up;
	struct tbuf *tp;
	int uid;

	for (tp=tbuf; tp<&tbuf[MAXTTY]; tp++)
		connup(tp, frtime);
	for (up=ubuf; up<&ubuf[MAXUSERS]; up++) {
		if (up->xname[0]) {
			uid = numuid(up->xname);
			aday[uid].uconn += up->uutime;
			up->uutime = 0;
		}
	}
}


struct passwd *pw;
struct passwd *getpwnam();

/*
* return the numeric userid for a given login name (0 if bad)
*/
numuid(name) char *name;
{
	int uid;

	if ((pw=getpwnam(name)) == 0)  return(0);
	uid = pw->pw_uid;
	return(
		(uid<0 || uid>MAXUSERS-1) ?
		0 : uid );
}


struct filsys sblock;
struct dinode  dinode[NINODE];
int fi;

/*
* countblk: counts the number of blocks owned by each user for all
*           currently mounted volumes.
*/
countblk()
{
	FILE *i;
	struct {
		char    m_name[32];
		char    m_dev[32];
	} mentry;

	chdir("/dev");
	i = fopen("/etc/mtab", "r");
	while(fread((char *)&mentry, sizeof mentry, 1, i) == 1) {
		if(mentry.m_dev[0] == 0) continue;
		du(mentry.m_dev);
	}
	fclose(i);
}

du(file)  char *file;
{
	int i, j, uid, ino;
	int nifiles;


	fi = open(file, 0);
	if(fi < 0)
		return;
	bread(1, (char *)&sblock, BSIZE);
	nifiles = (sblock.s_isize-2)*INOPB;
	ino = 0;
	for (i=0; ino<nifiles; i+= NINODE/INOPB) {
		bread(i+2, (char *)dinode, sizeof(struct dinode)*NINODE);
		for (j=0; j<NINODE && ino<nifiles; j++) {
			ino++;
			uid = dinode[j].di_uid;
			if (uid<0 || uid>MAXUSERS-1) uid = 0;
			if (dinode[j].di_size>0)
                                blocks[uid] += (dinode[j].di_size+BSIZE-1)/BSIZE;
		}
	}
	close(fi);
}


/*
* bread: reads a full block from the given file
*/
bread(bno, buf, size)
char *buf;
{
	int n;
	extern errno;

	seek(fi, bno*BSIZE, 0);
	if((n=read(fi, buf, size)) != size) {
		printf("read error %d\n", bno);
		printf("count = %d; errno = %d\n", n, errno);
		exit(1);
	}
}


/*
* blkload: loads the "aday" structure with user block counts.
*/
blkload()
{
	int n;

	for (n=0; n<MAXUSERS; n++)
		aday[n].ublk = blocks[n];
}


/*
* zeroout: fills the buffer pointed to with zeros
*          NOTE: this version assumes that the buffer consists of an
*          integral number of fullwords.
*/
zeroout(s, len) int *s;
{
	register i;

	for (i=0; i<(len/BYTEPWD); i++)
		*s++ = 0;
}


/*
* enter: finds a place for a command in the "cmds" table.
*/
enter(cmd) char *cmd;
{
	int i, j, k;

	i = k = 0;
	for (j=0; j<8; j++)
		i = i*7 + cmd[j];
	if (i<0)
		i = -i;
	for (i %= MAXCMDS; cmds[i].name[0]; i = (i+1)%MAXCMDS) {
		if (strcmp(cmd, cmds[i].name)==0)
			goto yes;
		if (++k > MAXCMDS) return(-1);
	}
	strcpy(cmds[i].name, cmd);

yes:    if ((i<0) || (i>MAXCMDS)) return(-1);
	return(i);
}
