#include "../h/param.h"
#include "../h/systm.h"
#include "../h/dir.h"
#include "../h/user.h"
#include "../h/proc.h"
#include "../h/text.h"
#include "../h/file.h"
#include "../h/inode.h"
#include "../h/buf.h"
#include "../h/stats.h"

#define EXITSVC 1

#define SQSIZE 0100	/* Must be power of 2 */
#define HASH(x)	(( (int) x >> 5) & (SQSIZE-1))
struct proc *slpque[SQSIZE];

/*
 * Give up the processor till a wakeup occurs
 * on chan, at which time the process
 * enters the scheduling queue at priority pri.
 * The most important effect of pri is that when
 * pri<=PZERO a signal cannot disturb the sleep;
 * if pri>PZERO signals will be processed.
 * Callers of this routine must be prepared for
 * premature return, and check that the reason for
 * sleeping has gone away.
 */
sleep(chan, pri)
caddr_t chan;
{
	register struct proc *rp;
	register h;

	rp = u.u_procp;
	rp->p_stat = SSLEEP;
	rp->p_wchan = chan;
	if (chan==0)
		panic("Sleeping on wchan 0");
	rp->p_pri = pri;
	h = HASH(chan);
	rp->p_link = slpque[h];
	slpque[h] = rp;
	if(pri > PZERO) {
		if(issig()) {
			rp->p_wchan = 0;
			rp->p_stat = SRUN;
			slpque[h] = rp->p_link;
			goto psig;
		}
		swtch();
		if(issig())
			goto psig;
	} else {
		swtch();
	}
	return;

	/*
	 * If priority was low (>PZERO) and
	 * there has been a signal,
	 * execute non-local goto to
	 * the qsav location.
	 * (see trap1/trap.c)
	 */
psig:
	if(u.u_intflg)
	        resume(u.u_qsav);
}

/*
 * Wake up all processes sleeping on chan.
 */
wakeup(chan)
register caddr_t chan;
{
	register struct proc *p, *q;
	register i;

	i = HASH(chan);
	p = slpque[i];
	q = NULL;
	while(p != NULL) {
		if(p->p_wchan==chan && p->p_stat!=SZOMB) {
			struct proc *sp;

			if (q == NULL)
				sp = slpque[i] = p->p_link;
			else
				sp = q->p_link = p->p_link;
			p->p_wchan = 0;
			setrun(p);
			p = sp;
			continue;
		}
		q = p;
		p = p->p_link;
	}
}

/*
 * when you are sure that it
 * is impossible to get the
 * 'proc on q' diagnostic, the
 * diagnostic loop can be removed.
 */
setrq(p)
register struct proc *p;
{
	register struct proc *q;

	for(q=runq; q!=NULL; q=q->p_link)
		if(q == p) {
			printf("proc on q\n");
			goto out;
		}
	p->p_link = runq;
	runq = p;
out:
	;
}

/*
 * Set the process running;
 * arrange for it to be swapped in if necessary.
 */
setrun(p)
register struct proc *p;
{
	register caddr_t w;

	if (p->p_stat==0 || p->p_stat==SZOMB)
		panic("Running a dead proc");
	/*
	 * The assignment to w is necessary because of
	 * race conditions. (Interrupt between test and use)
	 */
	if (w = p->p_wchan) {
		wakeup(w);
		return;
	}
	p->p_stat = SRUN;
	setrq(p);
	if(p->p_pri < curpri)
		runrun++;
}

/*
 * Set user priority.
 * The rescheduling flag (runrun)
 * is set if the priority is better
 * than the currently running process.
 */
setpri(pp)
register struct proc *pp;
{
	register p;

	p = (pp->p_cpu & 0377)/16;
	p += PUSER + pp->p_nice - NZERO;
	if(p > 127)
		p = 127;
	if(p < curpri)
		runrun++;
	pp->p_pri = p;
	return(p);
}

/*
 * The main loop of the scheduling (swapping)
 * process.
 * For now, do nothing.
 */
sched()
{
	for(;;)
		sleep((caddr_t)-1, PZERO);
}

/*
 * put the current process on
 * the Q of running processes and
 * call the scheduler.
 */
qswtch()
{

	setrq(u.u_procp);
	swtch();
}

/*
 * This routine is called to reschedule the CPU.
 * if the calling process is not in RUN state,
 * arrangements for it to restart must have
 * been made elsewhere, usually by calling via sleep.
 */
swtch()
{
	register n;
	register struct proc *p, *q, *pp, *pq;
	static struct user *oldu;
	cpu_t t, cpuswap();

	stats.s_swtchs++;
        t = cpuswap(0, T_HIGH);
        u.u_stime += t;
        u.u_procp->p_time += t;
        stats.s_stime += t;
        stats.s_ttime += t;
	/*
	 * If not the idle process, resume the idle process.
	 */
	if (u.u_procp != &proc[0]) {
		if (save(u.u_rsav)) {
			sureg();
			return;
		}
		/* Free U struct for dying process */
		if(u.u_procp->p_stat == SZOMB) {
			oldu = uptr;
                        stats.s_svc[EXITSVC].s_calls++;
                        stats.s_svc[EXITSVC].s_cputime += t;
		} else
			oldu = NULL;
		uptr = proc[0].p_addr;
		resume(u.u_qsav);
	}
	/*
	 * The first save returns nonzero when proc 0 is resumed
	 * by another process (above); then the second is not done
	 * and the process-search loop is entered.
	 *
	 * The first save returns 0 when swtch is called in proc 0
	 * from sched().  The second save returns 0 immediately, so
	 * in this case too the process-search loop is entered.
	 * Thus when proc 0 is awakened by being made runnable, it will
	 * find itself and resume itself at rsav, and return to sched().
	 */
	if (save(u.u_qsav)==0 && save(u.u_rsav))
		return;
	if(oldu) {
		freepag((caddr_t)oldu);
		oldu = NULL;
	}
	/*
	 * Open and close interrupt window
	 */
	window();

loop:
	runrun = 0;
	pp = NULL;
	q = NULL;
	n = 128;
	/*
	 * Search for highest-priority runnable process
	 */
	for(p=runq; p!=NULL; p=p->p_link) {
		if(p->p_stat==SRUN) {
			if(p->p_pri < n) {
				pp = p;
				pq = q;
				n = p->p_pri;
			}
		}
		q = p;
	}
	/*
	 * If no process is runnable, idle.
	 */
	p = pp;
	if(p == NULL) {
		t = cpuswap(2, T_HIGH);
		stats.s_swtime += t;
		stats.s_ttime += t;
		idle();
		goto loop;
	}
	q = pq;
	if(q == NULL)
		runq = p->p_link;
	else
		q->p_link = p->p_link;
	curpri = n;
	/*
	 * Address new user structure and return to process
	 */
	uptr = p->p_addr;
        t = cpuswap(0, T_HIGH);
        stats.s_swtime += t;
        stats.s_ttime += t;
	resume(u.u_rsav);
}

/*
 * Create a new process-- the internal version of
 * sys fork.
 * It returns 1 in the new process, 0 in the old.
 */
newproc()
{
	struct user *newu;
	struct proc *p, *up;
	register struct proc *rpp, *rip;
	register n;

	p = NULL;
	/*
	 * First, just locate a slot for a process
	 * and copy the useful info from this process into it.
	 * The panic "cannot happen" because fork has already
	 * checked for the existence of a slot.
	 */
retry:
	mpid++;
	if(mpid >= 30000) {
		mpid = 0;
		goto retry;
	}
	for(rpp = &proc[0]; rpp < &proc[NPROC]; rpp++) {
		if(rpp->p_stat == NULL && p==NULL)
			p = rpp;
		if (rpp->p_pid==mpid || rpp->p_pgrp==mpid)
			goto retry;
	}
	if ((rpp = p)==NULL)
		panic("no procs");

	/*
	 * make proc entry for new proc
	 */

	rip = u.u_procp;
	up = rip;
	rpp->p_stat = SRUN;
	rpp->p_clktim = 0;
	rpp->p_flag = SLOAD;
	rpp->p_uid = rip->p_uid;
	rpp->p_pgrp = rip->p_pgrp;
	rpp->p_nice = rip->p_nice;
	rpp->p_textp = rip->p_textp;
	rpp->p_pid = mpid;
	rpp->p_ppid = rip->p_pid;
	rpp->p_time = 0;
	rpp->p_cpu = 0;

	/*
	 * make duplicate entries
	 * where needed
	 */

	for(n=0; n<NOFILE; n++)
		if(u.u_ofile[n] != NULL)
			u.u_ofile[n]->f_count++;
	if(up->p_textp != NULL) {
		up->p_textp->x_count++;
	}
	u.u_cdir->i_count++;
	if (u.u_rdir)
		u.u_rdir->i_count++;
	/*
	 * When the resume is executed for the new process,
	 * here's where it will resume.
	 */
	if (save(u.u_rsav)) {
		sureg();
		return(1);
	}
	/*
	 * Create (by copying) the new process
	 */
	newu = (struct user *)getpage();
	copypag((caddr_t) &u, (caddr_t) newu);
	newu->u_segtab = NULL;
	newu->u_dseg.u_pages = newu->u_sseg.u_pages = NULL;
	newu->u_dseg.u_size = newu->u_sseg.u_size = 0;
	sbreak1(u.u_dseg.u_size, &newu->u_dseg);
	copysgs(&u.u_dseg, &newu->u_dseg);
	grow1(u.u_sseg.u_size, &newu->u_sseg);
	copysgs(&u.u_sseg, &newu->u_sseg);
	mkstab(newu, 0);
	fixstak(newu);
	newu->u_procp = rpp;
	newu->u_chreg = 1;
	rpp->p_addr = newu;
	setrq(rpp);
	return(0);
}
 
/*
 * Fixstak modifies addresses saved on stack (saved r12 and r13)
 * when stack is moved.  This makes the assumption that the
 * calling routine has done a save(newu->u_rsav).
 * This routine is inherently calling sequence dependent and ugly.
 */
fixstak(newu)
struct user *newu;
{
        register int diff, *p;
 
        diff = (int)newu - (int)uptr;
        newu->u_ar0 += diff / sizeof(int);
        newu->u_rsav[12-2] += diff;
        newu->u_rsav[13-2] += diff;
        p = (int *)newu->u_rsav[12-2];    /* r12 */
        while(p[12]) {      /* end condition guaranteed in zero.s */
                p[12] += diff;
                p[13] += diff;
                p = (int *)p[12];
        }
        p[13] += diff;
}
