MODULE lifegame;
CONST n = 30; { no. of cells }

DEVICE MODULE gt40[4];
DEFINE dispoff, display;

VAR pic: integer;
    dispoff, continue: signal;

PROCEDURE display(newpic: integer);
BEGIN
  pic := newpic;
  send(continue)
END display;

PROCESS screen[320B];
  VAR dpc[172000B]: integer;
BEGIN
  wait(continue);
  LOOP
    dpc := pic;
    doio;
    IF awaited(dispoff)
    THEN send(dispoff); wait(continue)
    END
  END
END screen;

BEGIN { gt40 }
  screen
END gt40;


MODULE life;
DEFINE reset, birth, death, alive, nextgeneration;
USE n;

TYPE position = ARRAY 0:n+1, 0:n+1 OF boolean;
VAR live: position;

PROCEDURE reset;
  VAR x, y: integer;
BEGIN
  x := 0;
  REPEAT
    y := 0;
    REPEAT live[x,y] := false; inc(y) UNTIL y>n+1;
    inc(x)
  UNTIL x>n+1
END reset;

PROCEDURE birth(x, y: integer);
BEGIN
  live[x, y] := true
END birth;

PROCEDURE death(x, y: integer);
BEGIN
  live[x, y] := false
END death;

PROCEDURE alive(x, y: integer): boolean;
BEGIN
  alive := live[x, y]
END alive;

PROCEDURE nextgeneration;
  VAR last: position;
      x, y, i: integer;
BEGIN
  last := live;
  x := 1;
  REPEAT
    y := 1;
    REPEAT
      i := 0;
      IF last[x-1,y-1] THEN inc(i) END;
      IF last[x-1,y]   THEN inc(i) END;
      IF last[x-1,y+1] THEN inc(i) END;
      IF last[x,y-1]   THEN inc(i) END;
      IF last[x,y+1]   THEN inc(i) END;
      IF last[x+1,y-1] THEN inc(i) END;
      IF last[x+1,y]   THEN inc(i) END;
      IF last[x+1,y+1] THEN inc(i) END;
      IF last[x,y] AND (i=2) OR (i=3)
      THEN birth(x, y)
      ELSE death(x, y)
      END;
      inc(y)
    UNTIL y>n;
    inc(x)
  UNTIL x>n
END nextgeneration;

BEGIN
  reset
END life;


MODULE cursor;
DEFINE cx, cy, setcursor;

VAR cx, cy: integer;

PROCEDURE setcursor(x, y: integer);
BEGIN
  cx := x; cy := y
END setcursor;

END cursor;




MODULE displayfile;
DEFINE drawcell, drawcursor, drawlife;
USE dispoff, display, n, alive, cx, cy;

TYPE dfile = RECORD
               cursor: ARRAY 1:5 OF integer;
               line: ARRAY 1:n OF
                     RECORD
                       head: ARRAY 1:4 OF integer;
                       cell: ARRAY 1:n OF char
                     END;
               stop: integer
             END;

VAR pic: ARRAY 0:1 OF dfile;
    on: integer;

CONST x0 = (2000B-14*n)/2;
      y0 = (1400B-14*n)/2;

PROCEDURE init(i: integer);
  VAR x, y: integer;
BEGIN
  WITH pic[i] DO
    cursor := (117124B, 14+x0, 14+y0, 100000B, 137B);
    y := 1;
    REPEAT
      WITH line[y] DO
        head := (114000B, 14+x0, 0, 100000B);
        head[3] := 14*y+y0;
        x := 1;
        REPEAT
          cell[x] := ' ';
          inc(x)
        UNTIL x>n
      END;
      inc(y)
    UNTIL y>n;
    stop := 173400B
  END
END init;

PROCEDURE drawcell(c: char);
BEGIN
  pic[on].line[cy].cell[cx] := c;
END drawcell;

PROCEDURE drawcursor;
BEGIN
  WITH pic[on] DO
    cursor[2] := 14*cx+x0;
    cursor[3] := 14*cy+y0
  END;
END drawcursor;

PROCEDURE drawlife;
  VAR x, y: integer;
BEGIN
  y := 1;
  REPEAT
    WITH pic[1-on].line[y] DO
      x := 1;
      REPEAT
        IF alive(x, y)
        THEN cell[x] := 'o'
        ELSE cell[x] := ' '
        END;
        inc(x)
      UNTIL x>n
    END;
    inc(y)
  UNTIL y>n;
  wait(dispoff);
  on := 1-on;
  display(adr(pic[on]))
END drawlife;

BEGIN
  init(0); init(1); on := 0
END displayfile;



DEVICE MODULE keyboard[4];
DEFINE chars, get;

VAR buffer: ARRAY 0:31 OF char;
    in, out, chars: integer;
    nonempty, nonfull: signal;

PROCEDURE get(VAR c: char);
BEGIN
  IF chars=0 THEN wait(nonempty) END;
  out := (out+1) MOD 32;
  c := buffer[out];
  dec(chars);
  send(nonfull)
END get;

PROCESS driver[60B];
  VAR ksr[177560B]: bits;
      kbr[177562B]: integer;
BEGIN
  LOOP
    IF chars=32 THEN wait(nonfull) END;
    ksr[6] := true; doio; ksr[6] := false;
    in := (in+1) MOD 32;
    buffer[in] := char(kbr MOD 128);
    inc(chars);
    send(nonempty)
  END
END driver;

BEGIN { keyboard }
  in := 0; out := 0; chars := 0;
  driver
END keyboard;


DEVICE MODULE timer[6];
DEFINE delay;

VAR tick: signal;

PROCEDURE delay(n: integer);
  VAR count: integer;
BEGIN
  count := n;
  REPEAT wait(tick); dec(count) UNTIL count<=0
END delay;

PROCESS clock[100B];
  VAR csr[177546B]: bits;
BEGIN
  LOOP csr[6] := true; doio; send(tick) END
END clock;

BEGIN
  clock
END timer;


PROCESS main;
  TYPE direction = (left, right, up, down);
  VAR d: direction;
      m: boolean;
      c: char;
  CONST intro = (117124B, 000400B, 000600B, 100000B,
                 067503B, 073556B, 074541B, 071447B,
                 063440B, 066541B, 020145B, 063157B,
                 066040B, 063151B, 000145B, 173400B);

  PROCEDURE move(newd: direction);
  BEGIN
    CASE newd OF
    left:  BEGIN
             IF cx>1 THEN setcursor(cx-1, cy) END
           END;
    right: BEGIN
             IF cx<n THEN setcursor(cx+1, cy) END
           END;
    up:    BEGIN
             IF cy<n THEN setcursor(cx, cy+1) END
           END;
    down:  BEGIN
             IF cy>1 THEN setcursor(cx, cy-1) END
           END
    END;
    d := newd; m := true; drawcursor
  END move;

  PROCEDURE home;
  BEGIN
    setcursor(1, 1);
    d := right; m := false;
    drawcursor
  END home;

BEGIN
  display(adr(intro[1])); delay(150);
  home; drawlife;
  LOOP
    get(c);
    CASE c OF
    10C:  BEGIN move(left); m := false END;
    30C:  BEGIN move(right); m := false END;
    32C:  BEGIN move(up); m := false END;
    13C:  BEGIN move(down); m := false END;
    ' ':  BEGIN
            IF m THEN move(d) END;
            m := true
          END;
    'o':  BEGIN
            IF m THEN move(d) END;
            birth(cx, cy); drawcell('o');
            m := true
          END;
    177C: BEGIN
            IF m THEN move(d) END;
            death(cx, cy); drawcell(' ');
            m := true
          END;
    'c':  BEGIN
            reset;
            drawlife;
            home
          END;
    'g':  BEGIN
            home;
            REPEAT
              nextgeneration;
              drawlife;
              IF chars<>0
              THEN REPEAT get(c) UNTIL (chars=0) OR (c='s')
              END
            UNTIL c='s'
          END
    END
  END
END main;

BEGIN { lifegame }
  main
END lifegame.
{
.bp
}
